/**
* \file: Session.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <systemd/sd-daemon.h>
#include <dipo_macros.h>
#include "Common.h"
#include "utils/Utils.h"
#include "Session.h"
#include "Server.h"
#include "control/IControlAdapterParameters.h"
#include "AirPlayUtils.h"
#include "AirPlayReceiverSessionPriv.h"
#include "utils/Statistics.h"

#include <sys/socket.h>
#include <sys/un.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <poll.h>

#define MAXMSG_SIZE  65535

using namespace std;

const char* CARPLAY_ENTITY_STRINGS[] = {
    "NA", "Car", "Mobile"
};
const char* CARPLAY_SPEECHMODE_STRINGS[] = {
    "NA", "None", "Recognizing", "Speaking"
};
const char* CARPLAY_TRANSFERTYPE_STRINGS[] = {
    "NA", "Take", "Untake", "Borrow", "Unborrow"
};
const char* CARPLAY_TRANSFERPRIORITY_STRINGS[] = {
    "NA", "NiceToHave", "UserInitiated"
};
const char* CARPLAY_CONSTRAINT_STRINGS[] = {
    "NA", "Anytime", "UserInitiated", "Never"
};
const char* CARPLAY_APPSTATE_STRINGS[] = {
    "NA", "True", "False"
};

#define CARPLAY_BOUND_CHECK(array,id) \
    (id < (sizeof(array)/ sizeof(char *)) ?  array[id] : "INVALID")


#define CARPLAY_MODESTATE_LOGFMT \
    "screen (%s), audio (%s), speech (%s, %s), phone (%s), navigation (%s)"
#define CARPLAY_MODESTATE_LOGARGS(a) \
    CARPLAY_BOUND_CHECK(CARPLAY_ENTITY_STRINGS,(a).screen), \
    CARPLAY_BOUND_CHECK(CARPLAY_ENTITY_STRINGS,(a).audio), \
    CARPLAY_BOUND_CHECK(CARPLAY_ENTITY_STRINGS,(a).speech.entity), \
    CARPLAY_BOUND_CHECK(CARPLAY_SPEECHMODE_STRINGS,(a).speech.mode), \
    CARPLAY_BOUND_CHECK(CARPLAY_ENTITY_STRINGS,(a).phone), \
    CARPLAY_BOUND_CHECK(CARPLAY_ENTITY_STRINGS,(a).navigation)

#define CARPLAY_RESOURCETRANSFER_LOGFMT \
    "%s, prio: %s, take constraint: %s, borrow constraint: %s"
#define CARPLAY_RESOURCETRANSFER_LOGARGS(a) \
    CARPLAY_BOUND_CHECK(CARPLAY_TRANSFERTYPE_STRINGS,(a).type), \
    CARPLAY_BOUND_CHECK(CARPLAY_TRANSFERPRIORITY_STRINGS,(a).priority), \
    CARPLAY_BOUND_CHECK(CARPLAY_CONSTRAINT_STRINGS,(a).takeConstraint), \
    CARPLAY_BOUND_CHECK(CARPLAY_CONSTRAINT_STRINGS,(a).borrowOrUnborrowConstraint)

#define CARPLAY_APPSTATES_LOGFMT \
    "speech: %s, phone: %s, navigation: %s"
#define CARPLAY_APPSTATES_LOGARGS(a) \
    CARPLAY_BOUND_CHECK(CARPLAY_SPEECHMODE_STRINGS,(a).speech), \
    CARPLAY_BOUND_CHECK(CARPLAY_APPSTATE_STRINGS,(a).phone), \
    CARPLAY_BOUND_CHECK(CARPLAY_APPSTATE_STRINGS,(a).navigation)

namespace adit { namespace carplay
{

class AutoMutex
{
public:
    AutoMutex(pthread_mutex_t * inMutex)
    {
        mutex = inMutex;
        pthread_mutex_lock(mutex);
    };

    ~AutoMutex()
    {
        pthread_mutex_unlock(mutex);
    };
private:
    pthread_mutex_t * mutex;
};


Session::Session()
{
    nightMode = NightMode_NotSupported;
    server = nullptr;
    config = nullptr;
    sessionInitialized = false;
    limitedUI = false;
    pthread_mutex_init(&closeLock, NULL);
    closed = false;
    phoneSessionRunning = false;
    supportLimitedUI = false;
    updateVehicleInfo = false;
    etcSupport = ETCInfo_NotSupported;
    nadSupport = NADInfo_NotSupported;
    Transport = CarPlay_Over_USB;
    sessionStop_fd = -1;
    iAP2OverCarPlay_fd = -1;
    audioChannel = nullptr;
}

Session::~Session()
{
    pthread_mutex_destroy(&closeLock);
}

struct sockaddr const * Session::getIPAddress(void)
{
  return (sockaddr *)&(GetReference()->peerAddr);
}


bool Session::Start(Server& inServer, IDynamicConfiguration& inConfig)
{
    server = &inServer;
    config = &inConfig;

    int count = GetInstanceCount();
    if (count < 10)
        LOGD_DEBUG((dipo, "currently %d Session objects alive", count));
    else
        LOG_WARN((dipo, "too many Session objects alive: %d, possible memory leak!", count));

    // init control channel
    if (!initControl())
    {
        LOG_ERROR((dipo, "failed to initialize session control"));
        return false;
    }

    // init input channel
    inputChannel.reset(new InputChannel(*this));
    if (!inputChannel->Initialize(config))
    {
        LOG_ERROR((dipo, "failed to initialize session input"));
        return false;
    }

    // Set to true to allow setNightMode
    sessionInitialized = true;

    Statistics::Instance().AddSession(this);
    sd_notifyf(0, "STATUS=Active session count=%d", Statistics::Instance().GetSessionCount());

    LOG_INFO((dipo, "CarPlay session object created"));
    return true;
}

void Session::Stop(bool inKeepObjectAlive)
{
    pthread_mutex_lock(&closeLock);
    closed = true;
    phoneSessionRunning = false;
    pthread_mutex_unlock(&closeLock);

    if(Transport == CarPlay_Over_WiFi )
    {
        if (sessionStop_fd >= 0)
        {
            // send stop event
            if (eventfd_write(sessionStop_fd, sessionStopEvent) != 0)
            {
                LOG_ERROR((dipo, "Failed to send stop event to iAP2OverCarPlay"));
            }
        }
        else
        {
            LOG_ERROR((dipo, "invalid sessionStop_fd"));
        }
    }


    Statistics::Instance().RemoveSession(this);
    if (Statistics::Instance().GetSessionCount() == 0)
        sd_notify(0, "STATUS=Waiting for session...");
    else
        sd_notifyf(0, "STATUS=Active sessions: %d", Statistics::Instance().GetSessionCount());

    // delete input channel and control adapter if existing
    inputChannel.reset(nullptr);
    controlAdapter.reset(nullptr);

    if(Transport == CarPlay_Over_WiFi )
    {
        pthread_join(iAP2WiFiCommThread_id, NULL);

        // close stop event fd
        if (sessionStop_fd >= 0)
        {
            close(sessionStop_fd);
            sessionStop_fd = -1;
        }
    }

    if (!inKeepObjectAlive)
    {
        if (server != nullptr)
        {
            // remove myself from server and destruct
            server->RemoveSession(*this);
        }

        LOG_INFO((dipo, "CarPlay session stopped"));
    }
    else
    {
        // If inKeepObjectAlive is set then the session will be stopped and resources are freed,
        // but the Session object will be kept alive.
        // This allows to capture further AirPlay notifications.

        // Generally only HandleSessionFinalized called from AirPlay will destruct the Session.
    }
}

AudioFormats Session::GetAudioFormats(AudioChannelType inChannelType,AudioType inStreamType)
{
    AudioFormats formats = 0;

	switch(inChannelType)
	{

		case AudioChannelType_Main:
			if(inStreamType == AudioStreamType_Default)
				formats = config->GetNumber("mainaudio-inout-default-formats",
						0);
			if(inStreamType == AudioStreamType_Media)
				formats = config->GetNumber("mainaudio-out-media-formats",
						0);
			if(inStreamType == AudioStreamType_Telephony)
				formats = config->GetNumber("mainaudio-inout-telephony-formats",
						0);
			if(inStreamType == AudioStreamType_SpeechRecognition)
				formats = config->GetNumber("mainaudio-inout-speech-formats",
						0);
			if(inStreamType == AudioStreamType_Alert)
				formats = config->GetNumber("mainaudio-out-alert-formats",
						0);
			break;
		case AudioChannelType_Alternate :
			if(inStreamType == AudioStreamType_Default)
				formats = config->GetNumber("alt-audio-out-default-formats",
						0);
			break;
		case AudioChannelType_Main_High :
			if(inStreamType == AudioStreamType_Media)
				formats = config->GetNumber("mainhighaudio-out-media-formats",
						0);
			break;
	}

    return formats;
}

AudioFormats Session::GetCompatibilityAudioFormats(bool inDirection, AudioChannelType inChannelType, AudioType inStreamType)
{
    AudioFormats formats = 0;

    switch(inChannelType)
    {
        case AudioChannelType_Main:
            if(inStreamType == AudioStreamType_Compatibility)
            {
                if(inDirection)
                    formats = config->GetNumber("mainaudio-in-compatibility-formats",
                        0);
                else
                    formats = config->GetNumber("mainaudio-out-compatibility-formats",
                    0);
            }
            break;
        case AudioChannelType_Alternate:
            if(inStreamType == AudioStreamType_Compatibility)
                    formats = config->GetNumber("alt-audio-out-compatibility-formats",
                            0);
            break;
        default:
            formats = 0;
            break;
    }
    return formats;
}

void Session::AudioPrepare(AudioChannelType inChannel, std::string inAudioType)
{
    if (controlAdapter != nullptr)
    {
        controlAdapter->OnAudioPrepare(inChannel, inAudioType);
    }
    else
    {
        LOG_ERROR((dipo, "cannot handle audio prepare, control adapter not found"));
    }
}

void Session::AudioStop(AudioChannelType inChannel)
{
    // fix for SWGIII-24156
    if(inChannel == AudioChannelType_Main_High)
        audioChannel = nullptr;

    if (controlAdapter != nullptr)
    {
        controlAdapter->OnAudioStop(inChannel);
    }
    else
    {
        LOG_ERROR((dipo, "cannot handle audio stop, control adapter not found"));
    }
}

void Session::VideoPlaybackStarted()
{
    if (controlAdapter != nullptr)
    {
        controlAdapter->OnVideoPlaybackStarted();
    }
    else
    {
        LOG_ERROR((dipo, "cannot handle video playback start, control adapter not found"));
    }
}

OSStatus Session::HandleSessionInitialized(AirPlayReceiverSessionRef inSession, void *inContext)
{
    (void)inContext;
    LOGD_DEBUG((dipo, "Session::HandleSessionInitialized ref=%p", inSession));

    return kNoErr;
}

void* Session::iAP2WiFiCommThread(void* exinf)
{
    (void)exinf;

    int rc = 0;
    struct sockaddr_un address;
    int socket_fd;
    socklen_t address_length;
    char message[MAXMSG_SIZE];
    fd_set read_fds;
    int fd_count=0;
    AirPlayReceiverSessionRef inSession = (AirPlayReceiverSessionRef)exinf;

    // set thread name
    prctl(PR_SET_NAME, "iAP2WiFiCommThread", 0, 0, 0);

    auto me = Session::Get(inSession);

    socket_fd = socket(PF_UNIX, SOCK_SEQPACKET, 0);
    if(socket_fd < 0)
    {
        LOG_ERROR((dipo, "socket() failed"));
        rc = -1;
    }
    else
    {
        LOGD_DEBUG((dipo, "Session::Sequential Socket Created ref=%p", inSession));
    }

    if(rc == 0)
    {
        unlink("/tmp/iAP2_Over_CarPlay");

        /* start with a clean address structure */
        memset(&address, 0, sizeof(struct sockaddr_un));

        address.sun_family = AF_UNIX;
        strncpy(address.sun_path, "/tmp/iAP2_Over_CarPlay", sizeof(address.sun_path));
        address.sun_path[sizeof(address.sun_path)-1] = '\0';

        if(bind(socket_fd, (struct sockaddr *) &address, sizeof(struct sockaddr_un)) != 0)
        {
            LOG_ERROR((dipo, "bind() failed"));
            rc = -1;
        }
        else
        {
            LOGD_DEBUG((dipo, "Session::Sequential Socket bind successfull ref=%p", inSession));
        }
    }

    if(rc == 0)
    {
        if(listen(socket_fd, 5) != 0)
        {
            LOG_ERROR((dipo, "listen() failed"));
            rc = -1;
        }
    }

    //Notify parent thread that socket creation done
    me->iAP2Mutex.lock();
    me->wiFiThreadCondVar.notify_one();
    me->iAP2Mutex.unlock();

    while(rc >= 0)
    {
        FD_ZERO(&read_fds);
        FD_SET(me->sessionStop_fd, &read_fds); //to detect session stop
        FD_SET(socket_fd, &read_fds);
        fd_count = max(socket_fd,me->sessionStop_fd);

        if(me->iAP2OverCarPlay_fd > -1)
        {
            FD_SET(me->iAP2OverCarPlay_fd, &read_fds);
            fd_count = max(me->iAP2OverCarPlay_fd, fd_count);
        }

        rc = select((fd_count + 1), &read_fds, NULL, NULL, NULL);
        if(rc > 0)
        {
            LOGD_DEBUG((dipo, "select() rc = %d", rc));
            if(FD_ISSET(me->sessionStop_fd, &read_fds))
            {
                LOGD_DEBUG((dipo, "Session stopped, exit from WiFithread"));
                break;
            }

            if(FD_ISSET(socket_fd, &read_fds))
            {
                me->iAP2OverCarPlay_fd = accept(socket_fd, (struct sockaddr *) &address, &address_length);
                rc = 1;
                LOGD_DEBUG((dipo, "Session::Sequential Socket Connection accepted ref=%p", inSession));
            }

            if(me->iAP2OverCarPlay_fd > -1)
            {
                if(FD_ISSET(me->iAP2OverCarPlay_fd, &read_fds))
                {
                    rc = recv(me->iAP2OverCarPlay_fd, message, MAXMSG_SIZE, 0);
                    if (rc < 0)
                    {
                        LOG_ERROR((dipo, "Error in receiving from Client"));
                        LOG_ERROR((dipo, "errno = %d, (%s), rc = %d", errno, strerror(errno), rc));
                        rc = -1;
                    }
                    else if(rc == 0)
                    {
                        LOGD_DEBUG((dipo, "peer has performed an orderly shutdown"));
                        rc = -1;
                    }
                    else
                    {
                        OSStatus error = kNoErr;
                        CFDataRef SendMsg = NULL;
                        LOGD_DEBUG((dipo, "recv() rc = %d", rc));
                        SendMsg = CFDataCreate(NULL, (const uint8_t *)&message, rc);
                        /* Send Message to the CarPlay - plugin-core */
                        error = AirPlayReceiverSessionSendiAPMessage(inSession, SendMsg, nullptr, nullptr);
                        LOGD_DEBUG((dipo, "AirPlayReceiverSessionSendiAPMessage()  returns error = %d", error));
                    }
                }
            }
        }
        else if(rc == 0)
        {
            /* select timeout */
        }
        else
        {
            LOG_ERROR((dipo, "errno = %d, (%s), rc = %d", errno, strerror(errno), rc));
        }
    }

    if(me->iAP2OverCarPlay_fd > -1)
    {
        LOGD_DEBUG((dipo, "Client Connection Closed"));
        rc = close(me->iAP2OverCarPlay_fd);
        me->iAP2OverCarPlay_fd = -1;
        if(rc == -1)
        {
            LOG_ERROR((dipo, "Error in closing iAP2OverCarPlay_fd, errno = %d (%s)", errno, strerror(errno)));
        }
    }


    if(socket_fd >= 0)
    {
        rc = close(socket_fd);
        if(rc == -1)
        {
            LOG_ERROR((dipo, "Error in closing socket_fd, errno = %d (%s)", errno, strerror(errno)));
        }
    }

    unlink("/tmp/iAP2_Over_CarPlay");

    return NULL;
}

void Session::HandleSessionStarted(AirPlayReceiverSessionRef inSession, void *inContext)
{
    (void)inContext;
    OSStatus err;
    CFTypeRef value;

    dipo_return_on_invalid_argument(dipo, inSession == nullptr);

    LOGD_DEBUG((dipo, "Session::HandleSessionStarted ref=%p", inSession));

    auto me = Session::Get(inSession);
    if (me == nullptr)
    {
        LOG_ERROR((dipo, "session not found %p for HandleSessionStarted", inSession));
        return; // ignore
    }

    // check transport type
    value = (CFNumberRef) AirPlayReceiverSessionCopyProperty(inSession, 0, CFSTR( kAirPlayProperty_TransportType ), NULL, &err );
    if( err == kNoErr && value )
    {
        uint32_t transportType;

        CFNumberGetValue( (CFNumberRef) value, kCFNumberSInt32Type, &transportType );
        if( NetTransportTypeIsWiFi( transportType ) )
        {
            me->Transport = CarPlay_Over_WiFi;

            /* create event fd to send session stop event to iAP2WiFiCommThread */
            me->sessionStop_fd = eventfd(0, 0);
            if(me->sessionStop_fd == -1)
            {
                LOGD_DEBUG((dipo, "Session: eventfd creation failed, cannot start iAP2OverCarPlay thread"));
            }
            else
            {
                /* Start iAP2 Over CarPlay */
                std::unique_lock<std::mutex> lock(me->iAP2Mutex);
                pthread_create(&me->iAP2WiFiCommThread_id, NULL, Session::iAP2WiFiCommThread, inSession);
                me->wiFiThreadCondVar.wait(lock);

                LOGD_DEBUG((dipo, "Session: WiFi Thread started"));
            }
        }
    }

    me->phoneSessionRunning = true;
    if(me->controlAdapter != nullptr)
    {
        me->controlAdapter->OnSessionStart(me->Transport);
    }
}

void Session::HandleSessionFinalized(AirPlayReceiverSessionRef inSession, void* inContext)
{
    (void)inContext;
    dipo_return_on_invalid_argument(dipo, inSession == nullptr);

    LOGD_DEBUG((dipo, "Session::HandleSessionFinalized ref=%p", inSession));
}

void Session::HandleModesChanged(AirPlayReceiverSessionRef inSession,
        const AirPlayModeState* inState, void* inContext)
{
    (void)inContext;
    dipo_return_on_invalid_argument(dipo, inSession == nullptr);
    dipo_return_on_invalid_argument(dipo, inState == nullptr);

    auto state = AirPlay2ModeState(*inState);
    LOGD_DEBUG((dipo, "mode changed: " CARPLAY_MODESTATE_LOGFMT,
            CARPLAY_MODESTATE_LOGARGS(state)));

    auto me = Session::Get(inSession);
    if (me != nullptr && me->controlAdapter != nullptr)
    {
        char buf[1024];
        snprintf(buf, sizeof(buf), CARPLAY_MODESTATE_LOGFMT, CARPLAY_MODESTATE_LOGARGS(state));
        Statistics::Instance().SetLatestModeChange(me, buf);

        me->controlAdapter->OnModesChanged(state);
    }
    else
    {
        LOG_ERROR((dipo, "cannot handle mode change, session not found %p", inSession));
    }
}

void Session::HandleRequestUI(AirPlayReceiverSessionRef inSession, CFStringRef inURL, void* inContext)
{
    (void)inContext;

    char url[1024];
    CFGetCString(inURL, url, sizeof(url));
    auto me = Session::Get(inSession);
    if (me != nullptr && me->controlAdapter != nullptr)
    {
        LOG_INFO((dipo, "Request native UI"));
        me->controlAdapter->OnRequestUI(url);
    }
    else
    {
        LOG_ERROR((dipo, "cannot handle UI request, session not found %p", inSession));
    }
}

OSStatus Session::HandleControl(AirPlayReceiverSessionRef inSession, CFStringRef inCommand,
        CFTypeRef inQualifier, CFDictionaryRef inParams, CFDictionaryRef* outParams,
        void* inContext)
{
    (void)inQualifier;
    (void)inParams;
    (void)outParams;
    (void)inContext;

    dipo_return_value_on_invalid_argument(dipo, inCommand == nullptr, kParamErr);

    char buffer[1024];
    CFGetCString(inCommand, buffer, sizeof(buffer));

    auto me = Session::Get(inSession);
    if (me == nullptr)
    {
        LOG_ERROR((dipo, "session not found %p for command %s", inSession, buffer));
        return kNoErr; // ignore
    }

    if (CFEqual(inCommand, CFSTR(kAirPlayCommand_DisableBluetooth)))
    {
        if (me != nullptr && me->controlAdapter != nullptr)
        {
            OSStatus error = kNoErr;
            CFDictionaryGetCString(inParams, CFSTR(kAirPlayKey_DeviceID), buffer, sizeof(buffer),
                    &error);
            if (error != kNoErr)
            {
               LOG_ERROR((dipo, "Bluetooth ID to disable not found"));
               return kUnknownErr;
            }
            else
            {
                LOGD_DEBUG((dipo, "Disable Bluetooth ID: %s", buffer));
                me->controlAdapter->OnDisableBluetooth(buffer);
            }
        }
    }
    else if (CFEqual(inCommand, CFSTR(kAirPlayCommand_SetUpStreams)))
    {
        OSStatus error = kNoErr;
        CFDictionaryGetCString(inParams, CFSTR(kAirPlayKey_OSBuildVersion), buffer, sizeof(buffer),
                &error);
        if (error == kNoErr)
        {
            LOG_INFO((dipo, "iOS build version: %s", buffer));
            if (me != nullptr)
                me->iOSBuildVersion = buffer;
        }
    }
    else if (CFEqual(inCommand, CFSTR(kAirPlayCommand_StartSession)))
    {
        if (me != nullptr)
        {
            me->phoneSessionRunning = true;
            me->controlAdapter->OnSessionStart(me->Transport);
        }
    }
    else if (CFEqual(inCommand, CFSTR(kAirPlayCommand_StopSession)))
    {
        if (me != nullptr)
        {
            me->controlAdapter->OnSessionEnd();
            me->Stop(false);
        }
    }
    else if (CFEqual(inCommand, CFSTR(kAirPlayCommand_iAPSendMessage)))
    {
        uint8_t *data;
        size_t   outLen;
        OSStatus outErr;

        data = CFDictionaryCopyData(inParams, CFSTR(kAirPlayKey_Data), &outLen, &outErr);
        LOG_INFO((dipo, "outLen: %zd, outErr = %d", outLen, outErr));
        if(outLen > 0)
        {
            int rc;

            rc = send(me->iAP2OverCarPlay_fd, data, outLen, 0);
            LOG_INFO((dipo, "send(): %d", rc));
        }
    }
    else if (CFEqual(inCommand, CFSTR(kAirPlayCommand_FlushAudio)))
    {
         LOG_INFO((dipo, "flush audio received"));
         if(me->audioChannel != nullptr)
         {
             me->audioChannel->Flush();
         }else
         {
             LOG_ERROR((dipo, "AudioChannel NULL"));
         }
    }
    else
    {
        LOG_ERROR((dipo, "Session::HandleControl: %s", buffer));
        return kNotHandledErr;
    }

    return kNoErr;
}

void Session::HandleDuckAudio(AirPlayReceiverSessionRef inSession, double inDurationSecs,
        double inVolume, void* inContext)
{
    (void)inContext;

    auto me = Session::Get(inSession);
    if (me != nullptr && me->controlAdapter != nullptr)
    {
        LOGD_DEBUG((dipo, "ramp volume %.6f s, %.6f", inDurationSecs, inVolume));
        me->controlAdapter->OnRampVolume(inVolume, inDurationSecs);
    }
    else
    {
        LOG_ERROR((dipo, "cannot handle duck audio, session invalid %p", inSession));
    }
}

void Session::HandleUnduckAudio(AirPlayReceiverSessionRef inSession, double inDurationSecs,
        void* inContext)
{
    (void)inContext;

    auto me = Session::Get(inSession);
    if (me != nullptr && me->controlAdapter != nullptr)
    {
        LOGD_DEBUG((dipo, "ramp volume %.6f s, %.6f", inDurationSecs, 1.0f));
        me->controlAdapter->OnRampVolume(1.0, inDurationSecs);
    }
    else
    {
        LOG_ERROR((dipo, "cannot handle unduck audio, session invalid %p", inSession));
    }
}

CFTypeRef Session::HandleCopyProperty(AirPlayReceiverSessionRef inSession, CFStringRef inProperty,
        CFTypeRef inQualifier, OSStatus* outErr, void* inContext)
{
    (void)inQualifier;
    (void)inContext;

    auto me = Get(inSession);
    auto config = me->config;

    (void)config;

    OSStatus status = kNoErr;
    CFTypeRef value = nullptr;

    char str[1024];
    CFGetCString(inProperty, str, 1024);

    if (false)
    {
    }
    else if (CFEqual(inProperty, CFSTR(kAirPlayProperty_Modes)))
    {
        ModeChanges changes;
        me->controlAdapter->OnGetCurrentResourceMode(changes);

        LOGD_DEBUG((dipo, "INFO current mode change: screen (" CARPLAY_RESOURCETRANSFER_LOGFMT ")",
                CARPLAY_RESOURCETRANSFER_LOGARGS(changes.screen)));
        LOGD_DEBUG((dipo, "INFO current mode change: audio (" CARPLAY_RESOURCETRANSFER_LOGFMT ")",
                CARPLAY_RESOURCETRANSFER_LOGARGS(changes.audio)));
        LOGD_DEBUG((dipo, "INFO current mode change: apps (" CARPLAY_APPSTATES_LOGFMT ")",
                CARPLAY_APPSTATES_LOGARGS(changes)));

        AirPlayModeChanges airplayModeChanges;
        ModeChanges2AirPlay(changes, airplayModeChanges);

        value = AirPlayCreateModesDictionary(&airplayModeChanges, nullptr, &status);
        if (status != kNoErr)
            LOG_ERROR((dipo, "failed to create modes dictionary (%d)", status));
    }
    else
    {
        LOG_WARN((dipo, "Session::CopyProperty: %s not handled", str));
        status = kNotHandledErr;
    }

    if (outErr)
    {
        *outErr = status;
    }

    return value;
}

OSStatus Session::HandleSetProperty(AirPlayReceiverSessionRef inSession, CFStringRef inProperty,
        CFTypeRef inQualifier, CFTypeRef inValue, void* inContext)
{
    (void)inSession;
    (void)inProperty;
    (void)inQualifier;
    (void)inValue;
    (void)inContext;

    char str[1024];
    CFGetCString(inProperty, str, 1024);

    LOG_WARN((dipo, "Session::HandleSetProperty: %s not handled", str));
    return kNotHandledErr;
}

void Session::ChangeResourceMode(const ModeChanges& inChanges)
{
    LOGD_DEBUG((dipo, "Request mode change: screen (" CARPLAY_RESOURCETRANSFER_LOGFMT ")",
            CARPLAY_RESOURCETRANSFER_LOGARGS(inChanges.screen)));
    LOGD_DEBUG((dipo, "Request mode change: audio (" CARPLAY_RESOURCETRANSFER_LOGFMT ")",
            CARPLAY_RESOURCETRANSFER_LOGARGS(inChanges.audio)));
    LOGD_DEBUG((dipo, "Request mode change: apps (" CARPLAY_APPSTATES_LOGFMT ")",
            CARPLAY_APPSTATES_LOGARGS(inChanges)));

    AutoMutex mutex = AutoMutex(&closeLock);

    AirPlayModeChanges changes;
    ModeChanges2AirPlay(inChanges, changes);

    if (!sessionInitialized || !phoneSessionRunning)
    {
        LOG_ERROR((dipo, "don't request change of resource mode before session is initialized " \
                "and running!"));
        return;
    }

    OSStatus status = AirPlayReceiverSessionChangeModes(reference,
            &changes,
            nullptr, // reason
            ChangeResourceModeCallback, // completion
            nullptr); // context
    if (status != kNoErr)
    {
        LOG_ERROR((dipo, "could not change resource mode (%d)", status));
        // errors occur only due to lost connection or invalid arguments, no actions required
    }
}

void Session::ChangeResourceModeCallback(OSStatus inStatus, CFDictionaryRef inResponse,
        void *inContext)
{
    (void)inResponse;
    (void)inContext;

    if (inStatus != kNoErr)
    {
        LOG_ERROR((dipo, "resource mode change was denied (%d)", inStatus));
    }
    else
    {
        LOGD_DEBUG((dipo, "resource mode change was accepted"));
    }
}

void Session::RequestUI(const string& inUrl)
{
    AutoMutex mutex = AutoMutex(&closeLock);
    LOG_INFO((dipo, "Request CarPlay UI"));

    if (!sessionInitialized || !phoneSessionRunning)
    {
        LOG_ERROR((dipo, "don't request CarPlay UI before session is initialized and running!"));
        return;
    }

    if (closed)
    {
        LOG_WARN((dipo, "sesion was closed while requesting CarPlay UI"));
        return;
    }

    CFStringRef str = CFStringCreateFromStdString(inUrl);
    if (str != nullptr)
    {
        OSStatus status = AirPlayReceiverSessionRequestUI(reference,
                str,
                nullptr, // completion
                nullptr); // context
        CFRelease(str);
        if (status != kNoErr)
        {
            LOG_ERROR((dipo, "could not request CarPlay UI (%d)", status));
            // errors occur only due to lost connection or invalid arguments, no actions required
        }
    }
}

void Session::RequestSiriAction(SiriAction inAction)
{
    AutoMutex mutex = AutoMutex(&closeLock);
    LOGD_DEBUG((dipo, "Request SiriAction (%d)", inAction));

    if (!sessionInitialized || !phoneSessionRunning)
    {
        LOG_ERROR((dipo, "don't request Siri Action before session is initialized and running!"));
        return;
    }

    if (closed)
    {
        LOG_WARN((dipo, "sesion was closed while requesting Siri"));
        return;
    }

    OSStatus status = AirPlayReceiverSessionRequestSiriAction(reference,
            SiriAction2AirPlay(inAction),
            nullptr, // completion
            nullptr); // context
    if (status != kNoErr)
    {
        LOG_ERROR((dipo, "could not request Siri (%d)", status));
        // errors occur only due to lost connection or invalid arguments, no actions required
    }
}

const string& Session::GetVersion(VersionEntity inEntity)
{
    // no close lock required

    static const string emptyString = "";

    switch (inEntity)
    {
        case VersionEntity_CarPlay:
            return CarPlaySourceVersion;
        case VersionEntity_ApplePluginCore:
            return CarPlayCommunicationsPluginVersion;
        case VersionEntity_PhoneiOS:
            return iOSBuildVersion;
        default:
            LOG_WARN((dipo, "GetVersion: invalid argument inEntity=%d", inEntity));
            return emptyString;
    }
}

void Session::SetBluetoothIDs(const std::list<std::string>& inDeviceIDs)
{
    // no close lock required

    for (auto item : inDeviceIDs)
        LOGD_DEBUG((dipo, "set Bluetooth ID: %s", item.c_str()));

    bluetoothIDs.assign(inDeviceIDs.begin(),inDeviceIDs.end());

    // store for later use in INFO message
}

void Session::SetNightMode(bool inActive)
{
    AutoMutex mutex = AutoMutex(&closeLock);
    LOGD_DEBUG((dipo, "set night mode (%d)", (int)inActive));

    if (closed)
    {
        LOG_WARN((dipo, "sesion was closed while setting night mode"));
        return;
    }

    nightMode = inActive ? NightMode_Enabled : NightMode_Disabled;

    if (sessionInitialized)
    {
        // if session is initialized send a SetNightMode command
        if (!phoneSessionRunning)
        {
            LOG_ERROR((dipo, "don't set night mode after init and before session is running!"));
            return;
        }

        OSStatus status = AirPlayReceiverSessionSetNightMode(reference, inActive,
                nullptr, // completion
                nullptr); // context
        if (status != kNoErr)
        {
            LOG_ERROR((dipo, "could not set night mode (%d)", status));
            // errors occur only due to lost connection or invalid arguments, no actions required
        }
    }
    else
    {
        // only store current night mode for later use in INFO message
    }
}

void Session::SetLimitedUI(bool inActive)
{
    AutoMutex mutex = AutoMutex(&closeLock);
    LOGD_DEBUG((dipo, "set limited UI (%d)", (int)inActive));

    if (closed)
    {
        LOG_WARN((dipo, "sesion was closed while setting night mode"));
        return;
    }

    limitedUI = inActive;

    if (sessionInitialized)
    {
        // if session is initialized send a SetLimitedUI command
        if (!phoneSessionRunning)
        {
            LOG_ERROR((dipo, "don't set limited UI after init and before session is running!"));
            return;
        }

        if (supportLimitedUI)
        {
            OSStatus status = AirPlayReceiverSessionSetLimitedUI(reference, inActive,
                    nullptr, // completion
                    nullptr); // context
            if (status != kNoErr)
            {
                LOG_ERROR((dipo, "could not set limited UI (%d)", status));
                // errors occur only due to lost connection or invalid arguments, no actions required
            }
        }
        else
        {
            LOG_ERROR((dipo, "cannot send SetLimitedUI UI command as limitedUI is not supported"));
        }
    }
    else
    {
        // only store current night mode for later use in INFO message
    }
}

void Session::UpdateVehicleInformation(VehicleInformation inVehicleInfo)
{
    (void)inVehicleInfo;
    AutoMutex mutex = AutoMutex(&closeLock);
    LOGD_DEBUG((dipo, "Update Vehicle Information"));

    if (closed)
    {
        LOG_WARN((dipo, "sesion was closed while sending UpdateVehicleInformation"));
        return;
    }

    etcSupport = inVehicleInfo.electronicTollSupport;
    nadSupport = inVehicleInfo.navigationAidedDrivSupport;

    if((etcSupport != ETCInfo_NotSupported) || (nadSupport != NADInfo_NotSupported))
    {
        LOG_INFO((dipo, "UpdateVehicleInformation: ETC Support: %d, NAD support: %d", etcSupport, nadSupport));

        updateVehicleInfo = true;

        if (sessionInitialized)
        {
            // if session is initialized send UpdateVehicleInformation command
            if (!phoneSessionRunning)
            {
                LOG_ERROR((dipo, "don't send UpdateVehicleInformation after init and before session is running!"));
                return;
            }

            auto updateVehicleInfo = CFDictionaryCreateMutable(NULL, 0, &kCFTypeDictionaryKeyCallBacks,
                    &kCFTypeDictionaryValueCallBacks);

            if(etcSupport != ETCInfo_NotSupported)
            {
                auto params = CFDictionaryCreateMutable(NULL, 0, &kCFTypeDictionaryKeyCallBacks,
                           &kCFTypeDictionaryValueCallBacks);

                CFDictionarySetValue( params, CFSTR( kAirPlayKey_Active ), (etcSupport == ETCInfo_Active) ? kCFBooleanTrue : kCFBooleanFalse );

                CFDictionarySetValue( updateVehicleInfo, CFSTR( kAirPlayVehicleInformation_ETC ), params );
                ForgetCF( &params );
            }

            if(nadSupport != NADInfo_NotSupported)
            {
                auto params = CFDictionaryCreateMutable(NULL, 0, &kCFTypeDictionaryKeyCallBacks,
                           &kCFTypeDictionaryValueCallBacks);

                CFDictionarySetBoolean( params, CFSTR( kAirPlayKey_Active ), (nadSupport == NADInfo_Active) ? true : false );

                CFDictionarySetValue( updateVehicleInfo, CFSTR( kAirPlayVehicleInformation_NavigationAidedDriving ), params );

                ForgetCF( &params );
            }

            OSStatus status = AirPlayReceiverSessionUpdateVehicleInformation(reference, updateVehicleInfo,
                    nullptr,  //inCompletion
                    nullptr); //inContext

            if (status != kNoErr)
            {
                LOG_ERROR((dipo, "could not send UpdateVehicleInformation command err:%d", status));
            }

            CFRelease(updateVehicleInfo);
        }
    }
    else
    {
        LOG_INFO((dipo, "UpdateVehicleInformation: both ETC and Navigation Aided Driving not supported."));
    }
}

// ====== private methods ======

bool Session::initControl()
{
    controlAdapter = move(createAdapter<IControlAdapter>("IControlAdapter"));
    if (controlAdapter != nullptr)
    {
        // TODO no cast, link directly to root config
        if(!controlAdapter->Initialize(*config, *this))
        {
            LOG_ERROR((dipo, "IControlAdapter initialization failed"));
            controlAdapter.reset(nullptr);
            return false;
        }
    }
    else
    {
        LOG_ERROR((dipo, "IControlAdapter not initialized"));
        return false;
    }

    // register control delegates
    AirPlayReceiverSessionDelegate delegate;
    AirPlayReceiverSessionDelegateInit(&delegate);
    delegate.initialize_f   = HandleSessionInitialized;
    delegate.started_f      = HandleSessionStarted;
    delegate.finalize_f     = HandleSessionFinalized;
    delegate.modesChanged_f = HandleModesChanged;
    delegate.requestUI_f    = HandleRequestUI;
    delegate.control_f      = HandleControl;
    delegate.duckAudio_f    = HandleDuckAudio;
    delegate.unduckAudio_f  = HandleUnduckAudio;
    delegate.copyProperty_f = HandleCopyProperty;
    delegate.setProperty_f  = HandleSetProperty;
    AirPlayReceiverSessionSetDelegate(reference, &delegate);
    return true;
}

template<class T> unique_ptr<T> Session::createAdapter(const std::string& inName)
{
    string impl = config->GetItem(inName, "");
    unique_ptr<T> ptr(static_cast<T*>(Factory::Instance()->Create(impl)));

    if (ptr == nullptr)
    {
        LOG_ERROR((dipo, "%s implementation %s not found!", inName.c_str(), impl.c_str()));
        return nullptr;
    }

    return move(ptr);
}

} } /* namespace adit { namespace carplay */
